/********************************************************************************/
/*    H   H Y   Y TTTTT EEEEE  CCC       HYTEC ELECTRONICS LTD                  */
/*    H   H  Y Y    T   E     C          5 Cradock Road,                        */
/*    HHHHH   Y     T   EEE   C          Reading, Berks.    Tel: 0118 9757770   */
/*    H   H   Y     T   E     C          RG2 0JT            Fax: 0118 9757566   */
/*    H   H   Y     T   EEEEE  CCC       Web: www.hytec-electronics.co.uk       */
/********************************************************************************/

/* $Author: cross $ */
/* $Date: 2005/02/02 10:57:49 $ */
/* $Id: drvHy8513.c,v 1.6 2005/02/02 10:57:49 cross Exp $ */
/* $Name:  $ */
/* $Revision: 1.6 $ */

/********************************************************************************/
/*                                                                              */
/*  drvHy8513.c - low level driver for the Hytec 8513 IP cards                  */
/*                                                                              */
/*  (C)2004 Hytec Electronics Ltd.                                              */
/*                                                                              */
/********************************************************************************/
/*                                                                              */
/*  Revision history: (comment and initial revisions)                           */
/*                                                                              */
/*  vers.       revised         modified by                                     */
/*  -----       -----------     -------------------------                       */
/*  See Above.  02-FEB-2005     Darrell Nineham Hytec Electronics Ltd.          */
/*                              Renamed Certain Functions to EPICS Standard.    */
/*                                                                              */
/*  1.5         14-DEC-2004     Darrell Nineham Hytec Electronics Ltd.          */
/*                              1.  Fixed Interrupt handling in ISR.            */
/*                              2.  Upgraded Report to display CRG_OVERFLOW.    */
/*                              3.  Hy8513_ai_driver() expanded to allow access */
/*                                  to 'capture registers'.                     */
/*                                                                              */
/*  1.4         18-NOV-2004     Darrell Nineham Hytec Electronics Ltd.          */
/*                              1.  'carrier' parameter incorrectly checked in  */
/*                                  Hy8513Configure() fix from Paul Hamadyk.    */
/*                              2.  Upgraded to Hytec coding standard.          */
/*                                                                              */
/*  1.3         10-SEP-2004     Darrell Nineham Hytec Electronics Ltd.          */
/*                              1.  Modified checkprom so that it will support  */
/*                                  IP modules that report Revision as 2202 or  */
/*                                  later.                                      */
/*                              2.  Removed some incorrect debug reporting.     */
/*                                  of the 8001 board.                          */
/*                                                                              */
/*  1.2         01-SEP-2004     Darrell Nineham Hytec Electronics Ltd.          */
/*                              1.  Modified checkprom so that it will support  */
/*                                  IP modules that report as Hy8513 or         */
/*                                  incorrectly as Hy8512 modules.              */
/*                              2.  Added TestCount routine which allows counts */
/*                                  to be simulated, allowing testing of        */
/*                                  counters without the need for actual        */
/*                                  encoders to be fitted.                      */
/*                              3.  Included new DryHy8513Report which more     */
/*                                  closely matches registers.                  */
/*                                                                              */
/*  1.1         25-JUN-2004     Paul Hamadyk DLS Ltd                            */
/*	                            Basic support for a Hytec Hy8513 IP module      */ 
/*                              (which may be reported as an Hy8512). Includes  */
/*                              interrupt code for testing only. It might be    */
/*                              used for the index pulse. Position registers    */
/*                              are set to zero on card insertion. Save and     */
/*                              restore the position using a database if you    */
/*                              need to remove a carrier card.                  */
/*                                                                              */
/********************************************************************************/

#include <vxWorks.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <vme.h>
#include <rebootLib.h>
#include <logLib.h>

#include "alarm.h"
#include "cvtTable.h"
#include "dbDefs.h"
#include "dbAccess.h"
#include "recSup.h"
#include "devSup.h"
#include "drvSup.h"
#include "dbScan.h"
#include "link.h"

#include "Hy8513.h"
#include "drvIpac.h"

#undef DEBUG
#define INTERRUPT

/* driver name */


/* model and revision can be overriden in the Configure function */

#define DRIVER		    "drvHy8513"
#define PROM_MODEL_0    0x8513
#define PROM_MODEL_1    0x8512
#define Hy8513REV	    0x2202      /* Earliest Usable Version - Hy8513 */
#define Hy8512REV	    0x2201      /* Earliest Usable Version - Hy8512 */

#define Hy8513MANID	    0x00800300
#define PROM_OFFS	    0x80
#define CREG_OFFS	    0x00
#define RAM_OFFS	    0x00

#define ROM_HEADER	    (PROM_OFFS + 0x0)
#define ROM_IDHI	    (PROM_OFFS + 0x6)
#define ROM_IDLO	    (PROM_OFFS + 0x8)
#define ROM_MODEL	    (PROM_OFFS + 0xA)
#define ROM_REV		    (PROM_OFFS + 0xC)
#define ROM_SERIAL	    (PROM_OFFS + 0x1A)

#define CRG_CONTROL	    (CREG_OFFS + 0)
#define CRG_ARM		    (CREG_OFFS + 1)
#define CRG_OVERFLOW	(CREG_OFFS + 2)
#define CRG_IRQMASK	    (CREG_OFFS + 3)
#define CRG_INTERVAL	(CREG_OFFS + 4)
#define CRG_BLOCK	    (CREG_OFFS + 5)
#define CRG_DAISY	    (CREG_OFFS + 6)
#define CRG_GATE	    (CREG_OFFS + 7)

#define CON_RESET	    (1<<1)
#define CON_SS		    (1<<2)
#define CON_EN		    (1<<3)
#define CON_TEST	    (1<<7)
#define CON_VECSHIFT	8

/* CSR - Control & Status Register Bit Mapping */
#define CSR_INDEX1	    0x0001
#define CSR_INDEX2	    0x0002
#define CSR_RESET 	    0x0002
#define CSR_INDEX3	    0x0004
#define CSR_INDEX4	    0x0008
#define CSR_SPARE1	    0x0010
#define CSR_SPARE2	    0x0020
#define CSR_SPARE3	    0x0040
#define CSR_TEST  	    0x0080
#define CSR_V0          0x0100
#define CSR_V1          0x0200
#define CSR_V2          0x0400
#define CSR_V3          0x0800
#define CSR_V4          0x1000
#define CSR_V5          0x2000
#define CSR_V6          0x4000
#define CSR_V7          0x8000


#define LO		        0
#define HI		        1

enum cardstate { REMOVED, INSERTED };


/* structure to support multiple IP modules */
struct cardinfo
{

    /* next structure */

	struct cardinfo *next;

    /* for interrupts */

	IOSCANPVT ioscanpvt;

    /* config details */

	unsigned short cardnum;			/* EPICS card number */
	unsigned short carrier;			/* carrier card number */
	unsigned short ipslot;			/* IP slot number */
	unsigned short init;			/* module initialised */
	unsigned short state;			/* insertion state */
	unsigned short intnum;			/* IP interrupt number */

	unsigned short *mem;			/* IP I/O */
	unsigned short *ram;			/* IP ram */
	
    /* writable I/O registers */

	unsigned short Control;
	unsigned short Arm;
	unsigned short Overflow;
	unsigned short IRQmask;
	unsigned short Interval;
	unsigned short Block;
	unsigned short Daisy;
	unsigned short Gate;

	/*	
	'Capture Registers' - Used to store the scalers count
	triggered by the relevant channel's limit switch interrupt.
	*/
	
	int iCaptureRegister[4];

    /* Storage for Hardware Model */

    unsigned short IP_Model;
    
}
cardinfo;


/* our single linked list of cardinfo structures */

static struct cardinfo *cardlist = NULL;


/*

  Interrupt Modes
  ---------------

  The Hy8513 can interrupts when a limit switch input is triggered
  (falling edge). The behavour at interrupt depends on the setting 
  of the below variable, iInterruptMode.

*/
#define INT_RESET_COUNTER		0	/* Reset Scaler Counter          */
#define INT_CAPTURE_COUNTER		1	/* Capture Scaler Count at Limit */

/* int iInterruptMode = INT_RESET_COUNTER; */
int iInterruptMode = INT_CAPTURE_COUNTER;

/* Function Declarations */

long DrvHy8513Report(int level);
static long Hy8513Init(void);
static int dumpcard(struct cardinfo *card);

struct drv_sup
{
	long		number;
	DRVSUPFUN	report;
	DRVSUPFUN	init;
};

struct drv_sup drvHy8513 =
{
	2,
	DrvHy8513Report,
	Hy8513Init
};


/**********************************************************

    NAME:  IO_REPORT (define of DrvHy8513Report)
    
    Description:
        
    Function - Display the IP card's registers and Counters 
    New Version more more closely matched to Hy8513.
	
            
    Parameters:
                
    Inputs:
    level - Display Level   1 -  Counters in Decimal 
                            0 -  Counters in Hex
    
    Outputs:        None.
    Return:         0 - Everything OK 
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 02-FEB-2005     DAN     1.6         Renamed Function to
                                     EPICS Standard.  
                                     
 07-DEC-2004     DAN     1.5         Added Display of 
                                     interrupt / overflow
                                     register. Also now
									 shows 'capture registers'

 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 01-SEP-2004     DAN     1.2         new DryHy8513Report 
                                     which more closely
                                     matches registers.
                                     Displays values in
                                     Hex or Decimal.
 
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
long IO_REPORT(int level)
{
	struct cardinfo *card = cardlist;
	unsigned short test, lo, hi, i;
	
	while (card != NULL)
	{
		/* Display General Purpose Registers */
		printf("\nCard - %d\n\n", card->cardnum);
		vxMemProbe((char *)(card->mem + CRG_CONTROL),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tControl  : %04x\n",(unsigned int)(card->mem + CRG_CONTROL), test);
		vxMemProbe((char *)(card->mem + CRG_ARM),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tArm      : %04x\n",(unsigned int)(card->mem + CRG_ARM), test);
		vxMemProbe((char *)(card->mem + CRG_OVERFLOW),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tInt      : %04x\n",(unsigned int)(card->mem + CRG_OVERFLOW), test);
        vxMemProbe((char *)(card->mem + CRG_IRQMASK),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tIRQmask  : %04x\n\n",(unsigned int)(card->mem + CRG_IRQMASK), test);
		
		/* Display Scaler Counters */
		printf("\tCounter  0\tCounter  1\tCounter  2\tCounter  3\n");
        
        printf("Addr -\t");
		for (i = 0; i < 4; i++)
		{
			printf("0x%08x\t",(unsigned int)(card->ram + RAM_OFFS + LO + 2 * i));
		}
		printf("\n");
		
        printf("Data -\t"); 
		for (i = 0; i < 4; i++)
		{
		    /* Read Counter */
			vxMemProbe((char *)(card->ram + RAM_OFFS + LO + 2 * i), READ, sizeof(short), (char *) &lo);
			vxMemProbe((char *)(card->ram + RAM_OFFS + HI + 2 * i), READ, sizeof(short), (char *) &hi);

            /* If Level = 1 Display Count in Decimal */
            if (level == 1)
            {
			    printf("%+10d\t",(hi << 16) + lo);
            }
            else
            {
			    printf("0x%08x\t",(hi << 16) + lo);
            }
		}
		printf("\n");
		printf("\n");
		printf("\n");

		/* Display 'Capture Registers' */ 
		printf("\tCapture  0\tCapture  1\tCapture  2\tCapture  3\n");
		
        printf("Data -\t"); 
		for (i = 0; i < 4; i++)
		{
            /* If Level = 1 Display Count in Decimal */
            if (level == 1)
            {
			    printf("%+10d\t", card->iCaptureRegister[i]);
            }
            else
            {
			    printf("0x%08x\t", card->iCaptureRegister[i]);
            }
		}
		printf("\n");
		
		puts("");
		
		card = card->next;
	}
	
	return (0);
}


/**********************************************************

    NAME:  DrvHy8513Report()
    
    Description:
        
    Function - Display the IP card's registers and Counters. 
            
    Parameters:
                
    Inputs:
    level - Display Level - Not Used ! 
    
    Outputs:        None.
    Return:         0 - Everything OK 
                            
    Notes:
    Original Version kept out of courtesy to Paul Hamadyk 
    incase needed for backward compatibilty.
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 XX-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 01-SEP-2004     DAN     1.2         Replaced with above
                                     version.
 
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
/*long DrvHy8513Report(int level)
{
	struct cardinfo *card = cardlist;
	unsigned short test, lo, hi, i;
	
	while (card != NULL)
	{
		printf("\nCard - %d\n", card->cardnum);
		vxMemProbe((char *)(card->mem + CRG_CONTROL),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tControl  : %04x\n",card->mem + CRG_CONTROL, test);
		vxMemProbe((char *)(card->mem + CRG_ARM),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tArm      : %04x\n",card->mem + CRG_ARM, test);
        vxMemProbe((char *)(card->mem + CRG_OVERFLOW),	READ,  sizeof(short), (char *) &test);
		printf("\tOverflow : %04x\n", test);
        vxMemProbe((char *)(card->mem + CRG_IRQMASK),	READ,  sizeof(short), (char *) &test);
		printf("0x%08x \tIRQmask  : %04x\n",card->mem + CRG_IRQMASK, test);
        vxMemProbe((char *)(card->mem + CRG_INTERVAL),	READ,  sizeof(short), (char *) &test);
		printf("\tInterval : %04x\n", test);
		vxMemProbe((char *)(card->mem + CRG_BLOCK),	READ,  sizeof(short), (char *) &test);
		printf("\tBlock    : %04x\n", test);
		vxMemProbe((char *)(card->mem + CRG_DAISY),	READ,  sizeof(short), (char *) &test);
		printf("\tDaisy    : %04x\n", test);
		vxMemProbe((char *)(card->mem + CRG_GATE),	READ,  sizeof(short), (char *) &test);
		printf("\tGate     : %04x", test);

		for (i = 0; i < 16; i++)
		{
			if ((i % 4) == 0) printf("\n\t");
			
			vxMemProbe((char *)(card->ram + RAM_OFFS + LO + 2 * i), READ, sizeof(short), (char *) &lo);
			vxMemProbe((char *)(card->ram + RAM_OFFS + HI + 2 * i), READ, sizeof(short), (char *) &hi);

			printf("%08x\t", (hi << 16) + lo);
		}
		
		puts("");
		
		card = card->next;
	}
	
	return (0);
}*/


/**********************************************************

    NAME:  Hy8513_ai_driver()
    
    Description:
        
    Function - Giving access to scaler counters and 'capture
	registers'.
            
    Parameters:
                
    Inputs:			cardnum - Which card to read.
                    signal	- Which signal to read...
								0-3 Scaler Counters 0-3.
								4-7 Capture Registers 0-3.
                    *parm	- Pointer to Parameter String.
                    *value	- Pointer to value in which to
							  return the requested data.
    
	Outputs:        None.
    
    Return:			S_drv_OK       - Updated OK
					S_drv_noDevice - Card NOT Found or 
									 Uninitialised or
									 NOT inserted or
									 Memory Write Fails
					S_drv_badParam - Invalid Signal Value.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 09-DEC-2004     DAN     1.5         Added access to 
									 'capture registers'.
                                     
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
long Hy8513_ai_driver(  const int cardnum,
                        const short signal,
                        const char *parm,
                        int *value)
{
	struct cardinfo *card = cardlist;
	unsigned short lo, hi;
	int status = 0;
	
	/* Search for the Passed card number */
	while ((card != NULL) && (card->cardnum != cardnum))
	{
		card = card->next;
	}
	
    /* Exit Returning Error if Card Wasn't Found */
	if (card == NULL)
	{
		return (S_drv_noDevice);
	}

    /* Exit Returning Error if Card Wasn't Initialised */
	if (!card->init)
	{
		return (S_drv_noDevice);
	}
	
    /* If Card Isn't Present */
	if (!ipmCarrierIsPresent(card->carrier))
	{
		if (card->state == INSERTED)
		{
			printf("%s: REMOVED card: %d  carrier: %d  slot: %d\n",
                                                            DRIVER,
                                                            card->cardnum,
                                                            card->carrier,
                                                            card->ipslot);
		}

		card->state = REMOVED;
		
		return (S_drv_noDevice);
	}

	/* If Card has Been Removed re-initialise it */
	if (card->state == REMOVED)
	{
		if (dumpcard(card) != 0)
		{
			return (S_drv_noDevice);
		}
	}

	/* Read from Actual Scaler counters */
	if		((signal >= 0) && (signal <= 3))
	{
		status += vxMemProbe((char *)(card->ram + RAM_OFFS + LO + 2 * signal), READ, sizeof(short), (char *) &lo);
		status += vxMemProbe((char *)(card->ram + RAM_OFFS + HI + 2 * signal), READ, sizeof(short), (char *) &hi);
		*value = ((int) hi << 16) + (int) lo;
	}
	/* Read from card structures 'capture registers' */
	else if ((signal >= 4) && (signal <= 7))
	{
		*value = card->iCaptureRegister[signal - 4];
	}
	/* Invalid 'signal' Requested */
	else
	{
		return (S_drv_badParam);
	}
	
	#ifdef DEBUG
	printf("\nHy8513_ai_driver - card %d  signal %d  parm %s  val %d\n", cardnum, signal, parm, *value);
	#endif	
	
	if (status == 0)
	{
		return (S_drv_OK);
	}
	else
	{
		return (S_drv_noDevice);
	}
}


/**********************************************************

    NAME:  Hy8513_ao_driver()
    
    Description:
        
    Function - Used to Overwrite / Update Scaler Counters.
            
    Parameters:
                
    Inputs:			cardnum - Which Card to Update.
                    signal  - 0-3 Which Counter to Update.
					          99  Special Case Increment 
							      All Counters.
                    *parm   - Pointer to String containing
							  configuration parameters.
                    *value  - Pointer to New Counter Value. 
    
	Outputs:        None.

    Return:			S_drv_OK       - Updated OK
					S_drv_noDevice - Card NOT Found or 
									 Uninitialised or
									 NOT inserted or
									 Memory Write Fails
					S_drv_badParam - Invalid Signal Value.
	
	
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
long Hy8513_ao_driver(  const int cardnum,
                        const short signal,
                        const char *parm,
                        int value)
{
	struct cardinfo *card = cardlist;
	int status = 0;
	
	/* Search for Passed Card Number */
	while ((card != NULL) && (card->cardnum != cardnum))
	{
		card = card->next;
	}

	
    /* If Card Wasn't Found Exit ! */
	if (card == NULL)
	{
		return (S_drv_noDevice);
	}


    /* If Card Wasn't Initialised Exit ! */
	if (!card->init)
	{
		return (S_drv_noDevice);
	}


    /* If Card Isn't Present */
	if (!ipmCarrierIsPresent(card->carrier))
	{
		if (card->state == INSERTED)
		{
			printf("%s: REMOVED card: %d  carrier: %d  slot: %d\n",
                                                            DRIVER,
                                                            card->cardnum,
                                                            card->carrier,
                                                            card->ipslot);
		}

		card->state = REMOVED;

		return (S_drv_noDevice);
	}

	
	/* If Card has Been Removed re-initialise it */
	if (card->state == REMOVED)
	{
		if (dumpcard(card) != 0)
		{
			return (S_drv_noDevice);
		}
	}

	#ifdef DEBUG
	printf("\nHy8513_ao_driver - card %d  signal %d  parm %s  val %d\n",
                                                                cardnum,
                                                                signal,
                                                                parm,
                                                                value);

	#endif

    /* write to scaler */
	if ((signal >= 0) && (signal <= 3))
	{
		unsigned short lo, hi, arm;
		
		lo = value & 0xFFFF;
		hi = (value >> 16) & 0xFFFF;

	    /* Disarm Scaler */
	
		arm = ~(1 << signal);
		vxMemProbe((char *)(card->mem + CRG_ARM), WRITE, sizeof(short), (char *) &arm);
		
	    /* write new value */
	
		status += vxMemProbe((char *)(card->ram + RAM_OFFS + LO + 2 * signal), WRITE, sizeof(short), (char *) &lo);
		status += vxMemProbe((char *)(card->ram + RAM_OFFS + HI + 2 * signal), WRITE, sizeof(short), (char *) &hi);

	    /* Re-arm Scaler */
	
		arm = card->Arm;
		vxMemProbe((char *)(card->mem + CRG_ARM), WRITE, sizeof(short), (char *) &arm);
	}
	else
		
    /* If Special Signal Value 99 then Increment All Scalers on the Card */
	if (signal == 99)
	{
		unsigned short test;
		
		test = card->Control | CON_TEST;
		
		status += vxMemProbe((char *)(card->mem + CRG_CONTROL), WRITE, sizeof(short), (char *) &test);
		
		test = card->Control;
		
		status += vxMemProbe((char *)(card->mem + CRG_CONTROL), WRITE, sizeof(short), (char *) &test);
	}
	
	/* Un-recognized Scaler Value - Exit with Error */
	else
	{
		return (S_drv_badParam);
	}
	
	
	/* Return Status - OK if No Errors updating memory */ 
	if (status == 0)
	{
		return (S_drv_OK);
	}
	else
	{
		return (S_drv_noDevice);
	}
}


/**********************************************************

    NAME:  Hy8513_mbbidirect_driver()
    
    Description:
        
    Function -  To Read the Control Register (CSR) to Get 
				Present State of Index Pulses.

            
    Parameters:
                
    Inputs:			cardnum - Which Card to Read.
                    signal  - Ignorred.
                    *parm   - Ignorred.
                    *value  - Pointer to Value where to 
							  put read index bits state. 
    
	Outputs:        None.

    Return:			S_drv_OK       - Updated OK
					S_drv_noDevice - Card NOT Found or 
									 Uninitialised or
									 NOT inserted or
									 Memory Write Fails
					S_drv_badParam - Invalid Signal Value.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
long Hy8513_mbbidirect_driver(  const int cardnum,
                                const short signal,
                                const char *parm,
                                int *value)
{
	struct cardinfo *card = cardlist;
	int status = 0;
	unsigned short bits;

	/* Search for passed Card Number */
	while ((card != NULL) && (card->cardnum != cardnum))
	{
		card = card->next;
	}

	
    /* If Card Wasn't Found Exit ! */
	if (card == NULL)
	{
		return (S_drv_noDevice);
	}


    /* If Card Wasn't Initialised */
	if (!card->init)
	{
		return (S_drv_noDevice);
	}

	
	/* If Card isnt Present */
	if (!ipmCarrierIsPresent(card->carrier))
	{
		if (card->state == INSERTED)
		{
			printf("%s: REMOVED card: %d  carrier: %d  slot: %d\n",
                                                        DRIVER,
                                                        card->cardnum,
                                                        card->carrier,
                                                        card->ipslot);
		}

		card->state = REMOVED;
		
		return (S_drv_noDevice);
	}


	/* If Card has Been Removed re-initialise it */
	if (card->state == REMOVED)
	{
		if (dumpcard(card) != 0)
		{
			return (S_drv_noDevice);
		}
	}


    /* Read the Control Register (CSR) to Get Present State of Index Pulses */
	status += vxMemProbe((char *)(card->mem + CRG_CONTROL), READ, sizeof(short), (char *) &bits);
	/* Mask off everyting except Index Bits and load into Records Value */
	*value = bits & 0xffff;
	
	#ifdef DEBUG
	printf("\nHy8513_mbbidirect_driver - card %d  signal %d  parm %s  val 0x%x\n", cardnum, signal, parm, *value);
	#endif	

	
	/* No Memory Read Failure return everthing OK */
	if (status == 0)
	{
		return (S_drv_OK);
	}
	else
	{
		return (S_drv_noDevice);
	}
}


/**********************************************************

    NAME:  checkprom()
    
    Description:
        
    Function - Just confirms PROM details correct i.e....  
	
    1. Check for "VITA4 " string. 
	2. Check for Hytec ID.
	3. Check for Valid IP Model (0x8512 or 0x8513).
	4. Display Debug Information.
	5. Return whether Check PROM Successful.

            
    Parameters:
                
    Inputs:         *card - Pointer to present Card Structure
    Outputs:        None.
    Return:         Bit Mapped as follows...

    0x0001 - "VITA4 " header NOT found.
    0x0002 - Correct Hytec ID NOT found.
    0x0004 - Valid PROM Model Number NOT found.
    0x0008 - Incorrect Revision Model.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 14-DEC-2004     DAN     1.5         Updated card structure
                                     with IP Model type as
                                     read from the PROM.
									 Tidied Model and Version
									 Checking.
                                     
 30-OCT-2004     DAN     1.4         Converted to Hytec
                                     Coding Standard.
 
 10-SEP-2004     DAN     1.3         Modified checkprom so
                                     that it will support
                                     IP modules that report
                                     Revision as 2202 or
                                     later.

 01-SEP-2004     DAN     1.2         Modified checkprom so
                                     that it will support
                                     IP modules that report
                                     as Hy8513 or incorrectly
                                     as Hy8512 modules.
 
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
static int checkprom(struct cardinfo *card)
{
	int ret = 0;
	unsigned int manid;
	unsigned short model, rev, serial, idlo, idhi;
	char header[] = "      ";
	
	/* Read Start of PROM Version Data */
	vxMemProbe((char *)(card->mem) + ROM_HEADER + 0, READ, sizeof(short), (char *) header + 0);
	vxMemProbe((char *)(card->mem) + ROM_HEADER + 2, READ, sizeof(short), (char *) header + 2);
	vxMemProbe((char *)(card->mem) + ROM_HEADER + 4, READ, sizeof(short), (char *) header + 4);

	/* If its NOT 'VITA4 ' update Error Return Code */
	if (strcmp("VITA4 ", header) != 0) ret += 1;

	/* Read manid and convert to preferred format */	
	vxMemProbe((char *)(card->mem) + ROM_IDLO, READ, sizeof(short), (char *) &idlo);
	vxMemProbe((char *)(card->mem) + ROM_IDHI, READ, sizeof(short), (char *) &idhi);
	manid = (idhi << 16) + idlo;

	vxMemProbe((char *)(card->mem) + ROM_MODEL,  READ, sizeof(short), (char *) &model);
	vxMemProbe((char *)(card->mem) + ROM_REV,    READ, sizeof(short), (char *) &rev);
	vxMemProbe((char *)(card->mem) + ROM_SERIAL, READ, sizeof(short), (char *) &serial);

    /* Update Card Structure with IP Model read from PROM - hopefully 8512 or 8513 */ 
    card->IP_Model = model;
    
	/* If its NOT the expected manid update Error Return Code */
	if (manid != Hy8513MANID) ret += 2;
	
	/* Check for the Correct 'Model Number in the PROM' */
    if (model == PROM_MODEL_0)
    {
		/* If not a sufficiently up to date version update Error Return Code */
		if (rev < Hy8513REV) 
		{
			ret += 8;
		}
		else
		{
			printf("DEBUG: Correct Hardware Found - %X\n",model);
		}
    }    
	/* Check for Usable Earlier Hardware Version */
    else if (model == PROM_MODEL_1)
    {
		/* If not a sufficiently up to date version update Error Return Code */
		if (rev < Hy8512REV) 
		{
			ret += 8;
		}
		else
		{
	        printf("WARNING: Usable Alternative Hardware Found - %X\n",model);
		}
    }    
    else
    {
        ret += 4;
    }    

	/* Display Information read from PROM */
	printf("%s: header: %6s  model: %04X  manid: %08X  rev: %04X  serial: %d\n",
                                                                    DRIVER,
                                                                    header,
                                                                    model,
                                                                    manid,
                                                                    rev,
                                                                    serial);
	
	return (ret);
}


/**********************************************************

    NAME:  dumpcard()
    
    Description:
        
    Function - Updates all the registers from the passed
	card structure...
	
	1.	Disarm the Interrupts.
	2.	Reload the Registers.
	3.	If No Errors in Updating Registers change card's 
		state to INSERTED.
	4.	Reset Capture Registers.
            
    Parameters:
                
    Inputs:			card - pointer to card structure for

    Outputs:        None.

    Return:         0		- All Registers written OK.
                    Non 0	- Errors writing registers.       
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 O6-DEC-2004     DAN     1.5         Disarm the Interrupts
                                     before writing the
                                     scalers registers.

 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
static int dumpcard(struct cardinfo *card)
{
	int status = 0;
	unsigned short iDisarmCode = 0;
	
	printf("%s: dumpcard  card: %d  carrier: %d  slot: %d\n", DRIVER, card->cardnum, card->carrier, card->ipslot);
	
	/* Disarm the Interrupts */
	status += vxMemProbe((char *)(card->mem + CRG_ARM),	    WRITE, sizeof(short), (char *) &iDisarmCode);
	status += vxMemProbe((char *)(card->mem + CRG_IRQMASK), WRITE, sizeof(short), (char *) &iDisarmCode);

	/* Reload the Registers */
	status += vxMemProbe((char *)(card->mem + CRG_CONTROL),	WRITE, sizeof(short), (char *) &(card->Control));
	status += vxMemProbe((char *)(card->mem + CRG_OVERFLOW),WRITE, sizeof(short), (char *) &(card->Overflow));
	status += vxMemProbe((char *)(card->mem + CRG_IRQMASK),	WRITE, sizeof(short), (char *) &(card->IRQmask));
	status += vxMemProbe((char *)(card->mem + CRG_INTERVAL),WRITE, sizeof(short), (char *) &(card->Interval));
	status += vxMemProbe((char *)(card->mem + CRG_BLOCK),	WRITE, sizeof(short), (char *) &(card->Block));
	status += vxMemProbe((char *)(card->mem + CRG_DAISY),	WRITE, sizeof(short), (char *) &(card->Daisy));
	status += vxMemProbe((char *)(card->mem + CRG_GATE),	WRITE, sizeof(short), (char *) &(card->Gate));
	status += vxMemProbe((char *)(card->mem + CRG_ARM),	    WRITE, sizeof(short), (char *) &(card->Arm));

	/* If No Errors in Updating Registers change card's state to INSERTED */ 
	if (status == 0)
	{
		card->state = INSERTED;
	}

	/* Reset Capture Registers */
	card->iCaptureRegister[0] = 0;
	card->iCaptureRegister[1] = 0;
	card->iCaptureRegister[2] = 0;
	card->iCaptureRegister[3] = 0;

	return (status);
}


/**********************************************************

    NAME:  drvISR()
    
    Description:
        
    Function - Interrupt Handler.
            
    Parameters:
                
    Inputs:         cardnum - the card which interrupted.
    Outputs:        None.
    Return:         None.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 06-DEC-2004     DAN     1.5         Implemented new 8513
                                     Interrupts.
 
 30-OCT-2004     DAN     1.4         Converted to Hytec
                                     Coding Standard.
 
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
#ifdef INTERRUPT
static void drvISR(int cardnum)
{
	struct cardinfo *card = cardlist;
	unsigned short InterruptCount;
	unsigned short InterruptScanBit = 0x00000001;
    unsigned short PresentInterruptStatus;

	unsigned short lo = 0;
	unsigned short hi = 0;
	unsigned short arm = 0;
	
    /* Search for the card causing the interrupt */
	while ((card != NULL) && (card->cardnum != cardnum))
	{
		card = card->next;
	}
	
    /* Card wasn't found - Exit */
	if (card == NULL)
	{
		return;
	}
	

    /* Disable the Interrupts */
	*(card->mem + CRG_IRQMASK) = 0x0000;
   
    /* Read the CSR */
	PresentInterruptStatus = *(card->mem + CRG_OVERFLOW);

    /* Inform EPICS */
	/* scanIoRequest(card->ioscanpvt); */

	/* Log Message concerning Interrupt */
	logMsg("%s: interrupt  card: %d - %04x  arm %04x\n",
                                            (int) DRIVER,
                                            card->cardnum,
                                            PresentInterruptStatus,
                                            *(card->mem + CRG_CONTROL),
                                            NULL,
                                            NULL);
	
    /* Scan to across Present Interrupt Status to check for each interrupt */
	for (InterruptCount = 0; InterruptCount < 4; InterruptCount++) 
	{
        /* Check each interrupt in turn, If Status indciates Interrupt has occured */
        if ((PresentInterruptStatus & InterruptScanBit) == InterruptScanBit)
		{
			/* If the Interrupt is to Reset the Actual Counter */
			if (iInterruptMode == INT_RESET_COUNTER)
			{
		
				/* Disarm Scaler */
                arm = 0;
				vxMemProbe((char *)(card->mem + CRG_ARM), WRITE, sizeof(short), (char *) &arm);
		
				/* Write New Value - i.e. Reset Count to Zero */
				vxMemProbe(	(char *)(card->ram + RAM_OFFS + LO + 2 * InterruptCount), 
							WRITE, 
							sizeof(short), 
							(char *) &lo);
				vxMemProbe(	(char *)(card->ram + RAM_OFFS + HI + 2 * InterruptCount), 
							WRITE, 
							sizeof(short), 
							(char *) &hi);

				/* Re-arm Scaler */
				arm = card->Arm;
				vxMemProbe((char *)(card->mem + CRG_ARM), WRITE, sizeof(short), (char *) &arm);
			}
			/* If the Interrupt is to Store the Present Count into a 'capture register' */
			else if (iInterruptMode == INT_CAPTURE_COUNTER)
			{	

				/* Read the relevant Scaler / Counter */
				vxMemProbe(	(char *)(card->ram + RAM_OFFS + LO + 2 * InterruptCount), 
							READ, 
							sizeof(short), 
							(char *) &lo);
				vxMemProbe(	(char *)(card->ram + RAM_OFFS + HI + 2 * InterruptCount), 
							READ, 
							sizeof(short), 
							(char *) &hi);

				/* Save it to relevant 'Capture Register' */
				card->iCaptureRegister[InterruptCount] = ((int) hi << 16) + (int) lo;
			}
		}

		/* Move bit mask to check next Interrupt */
		InterruptScanBit <<= 1;
	}

    /* Clear All the Interrupts */
	*(card->mem + CRG_OVERFLOW) = 0x0000;

    /* Re-enable the Interrupts */
	*(card->mem + CRG_IRQMASK) = 0x000F;
}
#endif


/**********************************************************

    NAME:  RebootHy8513
    
    Description:
        
    Function - reset the cards before reboot
            
    Parameters:
                
    Inputs:
    Outputs:        None.
    Return:         
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 XX-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
int RebootHy8513(void)
{
	struct cardinfo *card = cardlist;

	while (card != NULL)
	{
		short reset = CON_RESET;
			
		vxMemProbe( (char *)(card->mem + CRG_CONTROL),
                    WRITE,
                    sizeof(short),
                    (char *) &reset);
		
		card = card->next;
	}
	
	return (0);
}


/**********************************************************

    NAME:  Hotswap()
    
    Description:
        
    Function - to handle hotswap events from carrier
            
    Parameters:
                
    Inputs:			carrier - Which carrier had the event
					event   - The event that occured i.e.
							  HS_Extract or HS_Insert.
    
	Outputs:        None.
    
	Return:			 0  Everything OK.
					-1  Card NOT Found.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
#if 0
static int Hotswap(unsigned short carrier, HS_event event)
{
	struct cardinfo *card = cardlist;

	/* Search for card with passed carrier id */ 
	while ((card != NULL) && (card->carrier != carrier))
	{
		card = card->next;
	}

	
    /* If card wasn't found Exit */
	if (card == NULL)
	{
		return (-1);
	}


	/* If Cards been Removed / Extracted */
	if (event == HS_Extract)
	{
		printf("ipac: carrier %d removed (card %d, slot %d)\n", card->carrier, card->cardnum, card->ipslot);
		
		card->state = REMOVED;
	}
	/* If Cards been Inserted */
	else
	if (event == HS_Insert)
	{
		struct cardinfo *card = cardlist;

		printf("ipac: carrier %d inserted (card %d, slot %d)\n", card->carrier, card->cardnum, card->ipslot);

		while ((card != NULL) && (card->carrier != carrier))
		{
			card = card->next;
		}
		
		if (card != NULL)
		{
			dumpcard(card);
		}
	}
	else
	{
		puts("ipac: unknown event");
	}
	

	/* Return Everything OK */
	return (0);
}
#endif


/**********************************************************

    NAME:  LIST (defined as Hy8513List)
    
    Description:

	Function - List Information about all cards initialised. 
            
    Parameters:
                
    Inputs:         None.
    Outputs:        None.
    Return:			0 - Everthing OK.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 02-FEB-2005     DAN     1.6         Renamed Function to
                                     EPICS Standard.  
                                     
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
long LIST(void)
{
	struct cardinfo *card = cardlist;

	while (card != NULL)
	{
	
		#ifdef INTERRUPT

		printf("%s:  card: %2d  carrier: %2d  ip: %d  int: %3d %sconfigured  %s\n",
                DRIVER,
                card->cardnum,
                card->carrier,
                card->ipslot,
                card->intnum,
                (card->init)  ? "    " : "not ",
                (card->state) ? "inserted" : "removed");
		
		#else

		printf("%s:  card: %2d  carrier: %2d  ip: %d  %sconfigured  %s\n",
                DRIVER,
                card->cardnum,
                card->carrier, 
                card->ipslot, 
                (card->init)  ? "    " : "not ",
                (card->state) ? "inserted" : "removed");

		#endif
		card = card->next;
	}
	
	return (0);
}


/**********************************************************

    NAME:  CONFIGURE (defined as Hy8513Configure)
    
    Description:
        
    Function - Configure Hy8513 from passed parameters.
	1. Check Parameters are valid, within acceptable range.
	2. Allocate structure and initialise with passed values.
	3. Setup reboot handler.
    4. Setup hot swap handler.

            
    Parameters:
                
    Inputs:

		cardnum	- card number as used in INP 
        carrier	- which VME Slot in the IOC
        ipslot	- which IP Slot on carrier card (0=A etc.)
        intnum	- which Interrupt Vector

    Outputs:        None.

    Return:	
	
		-1	- An Error conferred during configuration.
		 0	- The configuration was successful.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 02-FEB-2005     DAN     1.6         Renamed Function to
                                     EPICS Standard.  
                                     
 18-NOV-2004     DAN     1.4         1. Fixed bug (found by
									 Paul Hamadyk) that wont
									 allow the valid carrier
									 value of 0.
									 2. Upgraded to Hytec
                                     coding standard.

  
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
int CONFIGURE(const int cardnum, 
              const int carrier, 
              const int ipslot, 
              const int intnum)
{
	struct cardinfo *card = cardlist;

	#ifdef INTERRUPT
	
	printf("%s:  card: %d  vme: %2d  ip: %d  int: %d\n",
                                                DRIVER,
                                                cardnum,
                                                carrier,
                                                ipslot,
                                                intnum);

	#else

	printf("%s:  card: %d  vme: %2d  ip: %d\n", 
                                                DRIVER,
                                                cardnum,
                                                carrier,
                                                ipslot);

	#endif

    /* Check Passed Parameters */

	if ((intnum < 0) || (intnum > 255))
	{
		printf("%s: Illegal interrupt vector.\n", DRIVER);
		
		return (-1);
	}
	
	if ((ipslot < 0) || (ipslot > 3))
	{
		printf("%s: Illegal ip slot number.\n", DRIVER);

		return (-1);
	}
	
	if ((carrier < 0) || (carrier > 21))
	{
		printf("%s: Illegal carrier number %d.\n", DRIVER, 
                                                   carrier);
		
		return (-1);
	}
		
	while (card != NULL)
	{
		if (card->cardnum == cardnum) 
		{
			printf("%s: Card %d already defined.\n", DRIVER, cardnum);
			return (-1);
		}
		else
		{
			if ((card->carrier == carrier) && (card->ipslot == ipslot))
			{
				printf("%s: VME and IP slot already defined.\n", DRIVER);
				return (-1);
			}
		}

		if ((card->intnum == intnum) && (intnum != 0))
		{
			printf("%s: Interrupt %d in use for card %d\n",
                                                    DRIVER,
                                                    intnum,
                                                    card->cardnum);
			
			return (-1);
		}
		
		card = card->next;
	}

    /* allocate structure and initialise */

	card = calloc(1, sizeof(struct cardinfo));
	
	if (card == NULL)
	{
		printf("%s: Memory allocation error.\n", DRIVER);
		return (-1);
	}
	
	card->cardnum	= cardnum;
	card->carrier	= carrier;
	card->ipslot	= ipslot;
	card->intnum	= intnum;
	card->init	    = 0;

	card->Control	= 0x0000;
	card->Overflow	= 0x0000;	/* reset overflow bits */
	card->IRQmask	= 0x0000;	/* interrupt on index */

	card->Interval	= 0x0000;	/* no interval timers */
	card->Block	    = 0x0000;	/* no scaler blocks */
	card->Daisy	    = 0x0000;	/* no 64 bit scalers */
	card->Gate	    = 0x0000;	/* no 10 MHz clock */
	/* card->Arm	    = 0x000F;	 arm 0, 1, 2, 3 - Hy8513  */
	card->Arm	    = 0xFFFF;	/* arm - Hy8512 */

	card->next = cardlist;	
	cardlist = card;
	
    /* reboot handler */

	if (rebootHookAdd(RebootHy8513) < 0)
	{
		printf("%s: rebootHookAdd() failed.\n", DRIVER);
		return (-1);
	}

	#if 0
    /* hot swap handler */

	if (ipmInstallHSCallBack(carrier, Hotswap) < 0)
	{
		printf("%s: ipmInstallHSCallBack() failed.\n", DRIVER);
		return (-1);
	}
	#endif
	
	return (0);
}


/**********************************************************

    NAME:   DRIVER_INIT (defined as Hy8513Init)
    
    Description:
        
    Function - Initialisation of Hardware.
            
    Parameters:
                
    Inputs:         None.
    Outputs:        None.
    Return:			0		- OK
					-ve		- Error
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 02-FEB-2005     DAN     1.6         Renamed Function to
                                     EPICS Standard.  
                                     
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
static long DRIVER_INIT(void)
{
	struct cardinfo *card = cardlist;
	short tmp;
	int status, status2, ret = 0;

    printf("Hy8513Init() called.\n");
	
	while (card != NULL)
	{
		/* If Card already initialised */
        if (card->init)
		{
			printf("%s: Card %d already initialised.\n",
                                                DRIVER,
                                                card->cardnum);
		}
		else
		{
			printf("\n%s: card %d\n", DRIVER, card->cardnum);
			
		    /* register the card in A16 address space */
			card->mem = (unsigned short *) ipmBaseAddr( card->carrier,
                                                        card->ipslot,
                                                        ipac_addrIO);

		    /* map the RAM */
			card->ram = (unsigned short *) ipmBaseAddr( card->carrier, 
                                                        card->ipslot,
                                                        ipac_addrMem);
	
			/* If either of the address spacing fails */
            if ((card->mem == NULL) || (card->ram == NULL))
			{
				printf("%s: ipmBaseAddr failed for card %d.\n",
                                                        DRIVER,
                                                        card->cardnum);
			}
			else
			{
				printf("%s: ipmBaseAddr(mem) 0x%x\n", DRIVER, (unsigned int) card->mem);
				printf("%s: ipmBaseAddr(ram) 0x%x\n", DRIVER, (unsigned int) card->ram);
			
			    /* verify that the card exists */
	
				status = vxMemProbe((char *)(card->mem), READ, sizeof(short), (char *) &tmp);
				if (status != 0)
				{
					printf("%s: Memory (mem) probe error for card %d.\n", DRIVER, card->cardnum);
				}
				
                status2 = vxMemProbe((char *)(card->ram), READ, sizeof(short), (char *) &tmp);
				if (status2 != 0)
				{
					printf("%s: Memory (ram) probe error for card %d.\n", DRIVER, card->cardnum);
				}

                status += status2;
			
				if (status != 0)
				{
					printf("%s: Memory probe error for card %d.\n", DRIVER, card->cardnum);
				}
				else
				{
					card->state = INSERTED;
					
				    /* check the ROM */
			
					status = checkprom(card);
				
				    /* configure the card */
			
					if (status == 0)
					{
					    /* reset card */
						const unsigned short reset = CON_RESET;
						vxMemProbe((char *)(card->mem + CRG_CONTROL), WRITE, sizeof(short), (char *) &reset);
										
					    /* init interrupts */

						#ifdef INTERRUPT
						if (card->intnum > 0)
    					{
							/* scanIoInit(&(card->ioscanpvt));*/
						
							if (ipmIntConnect(  card->carrier,
                                                card->ipslot,
                                                card->intnum,
                                                drvISR,
                                                card->cardnum) != 0)
							{
								printf("%s: ipmIntConnect failed.\n", DRIVER);
								ret--;
							}
							else
							{
								/* Load Vector Interrupt in the top bits of the CSR */
                                card->Control |= (card->intnum << CON_VECSHIFT);
                                
                                /* Enable Interrupts */
								card->IRQmask  = 0x000F; 
                                card->Arm      = 0x000F;
								
								/* Clear Overflow / limit Interrupt Flags */
								card->Overflow = 0x0000;
								
								ipmIrqCmd(card->carrier, card->ipslot, 0, ipac_irqEnable);
							
							/*	printf(     "%s: Carrier interrupt level %d, %d\n", 
                                            DRIVER,
                                            ipmIrqCmd(card->carrier, card->ipslot, 0, ipac_irqGetLevel),
                                            ipmIrqCmd(card->carrier, card->ipslot, 1, ipac_irqGetLevel));
                            */
							}
						} 	    
						#endif						
					    
						/* init card with defaults */

						dumpcard(card);
						
						printf("%s: card configured.\n", DRIVER);
						
						card->init = 1;
					}
					else
					{
						printf("configure failed %d\n", status);
						ret--;
					}
                    
				}   /* End of Checking Memory OK */ 
                
			}   /* End of Intialises Memory OK */
            
		}   /* End of Card Not Initialised */
		
		card = card->next;
	}
	
	return (ret);
}


/**********************************************************

    NAME:  Hy8513getioscanpvt()
    
    Description:
        
    Function - Load the passed 'scanpvt' from the one in the
	card structure indicated by the passed card number.
            
    Parameters:
                
    Inputs:			cardnum
					*scanpvt

    Outputs:        None.

    Return:         -1 - Failed to find Card
					 0 - Everything OK
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 10-DEC-2004     DAN     1.5         Changed function name
                                     to EPICS standard.
                                     
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
int Hy8513getioscanpvt(const int cardnum, IOSCANPVT *scanpvt)
{
	struct cardinfo *card = cardlist;
    
	/* Search for the card causing the interrupt */
	while ((card != NULL) && (card->cardnum != cardnum))
	{
		card = card->next;
	}
	
    /* Card wasn't found - Exit */
	if (card == NULL)
	{
		printf("%s: Hy8513getioscanpvt: cardnum %d not found!\n",
                                                    DRIVER,
                                                    cardnum);
		return (-1);
	}

	/* Load 'scanpvt' from the one in the card's structure */ 
	*scanpvt = card->ioscanpvt;
	
	return (0);
}


/**********************************************************

    NAME:  ResetCard()
    
    Description:
        
    Function - Called to Reset Card.
            
    Parameters:
                
    Inputs:			cardnum - which card is be reset.

	Outputs:        None.
    
	Return:         -1		- Failed to find Card.
					 0		- Everything OK.
					Other	- Memory Write Failed.
                            
    Notes:
                            
    Programmer(s):  Darrell Nineham
                    Paul Hamadyk

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 XX-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 25-JUN-2004     PH      1.1         Intial Version by
                                     Paul Hamadyk Checked
                                     in by Darrell Nineham.

**********************************************************/
int ResetCard(const int cardnum)
{
	struct cardinfo *card = cardlist;
	unsigned short status = 0;
	
	/* Search for the passed card number */
	while ((card != NULL) && (card->cardnum != cardnum)) card = card->next;
	
	/* If card cant be found */
	if (card == NULL)
	{
		puts("No card");
		return (-1);
	}
	
	/* Re write Card */
	status += vxMemProbe((char *)(card->mem + CRG_CONTROL),	WRITE, sizeof(short), (char *) &(card->Control));
	status += vxMemProbe((char *)(card->mem + CRG_OVERFLOW),WRITE, sizeof(short), (char *) &(card->Overflow));
	status += vxMemProbe((char *)(card->mem + CRG_IRQMASK),	WRITE, sizeof(short), (char *) &(card->IRQmask));
	status += vxMemProbe((char *)(card->mem + CRG_INTERVAL),WRITE, sizeof(short), (char *) &(card->Interval));
	status += vxMemProbe((char *)(card->mem + CRG_BLOCK),	WRITE, sizeof(short), (char *) &(card->Block));
	status += vxMemProbe((char *)(card->mem + CRG_DAISY),	WRITE, sizeof(short), (char *) &(card->Daisy));
	status += vxMemProbe((char *)(card->mem + CRG_GATE),	WRITE, sizeof(short), (char *) &(card->Gate));
	status += vxMemProbe((char *)(card->mem + CRG_ARM),	    WRITE, sizeof(short), (char *) &(card->Arm));
	
	return (status);
}


/**********************************************************

    NAME:  TestCount()
    
    Description:
        
    Function - Test Routine.
    This is a test routine using the test bit in the CSR
    register.  All counters on the card are 'clocked'
    the passed 'count' number of times. 
            
    Parameters:
                
    Inputs:
    int cardnum - Which Card is to be tested.
    int count   - How many 'clocks' to apply to counters.
    
    Outputs:        None.
    
    Return:         None.
    -1 - Cannot find matching card.
     0 - Test Count Completed successfully.
                            
    Notes: The count direction is dependant on the
    hardware inputs.
                            
    Programmer(s):  Darrell Nineham

**********************************************************
                                                                                                
 Revision History
 ----------------
                                                                                
 Date            By      Version     Descripition
 ----            --      -------     ------------
 10-DEC-2004     DAN     1.5         Initialised CSR with 
									 Interrupt Vector.
                                     
 30-OCT-2004     DAN     1.4         Upgraded to Hytec
                                     coding standard.
                                     
 01-SEP-2004     DAN     1.2         Added TestCount routine
                                     which allows counts to
                                     be simulated, allowing
                                     testing of counters
                                     without the need for
                                     actual encoders to be
                                     fitted.                   

**********************************************************/
int TestCount(const int cardnum, int count)
{
	struct cardinfo *card = cardlist;
	unsigned short CSRvalue;
    int iClockCount;
	
	/* Search through all card structures for passed card number */ 
    while ((card != NULL) && (card->cardnum != cardnum)) card = card->next;
	
    /* If at end of the card list without finding a match */
	if (card == NULL)
	{
		puts("No card");
		return (-1);
	}

	/* Load CSR with Interrupt Value */
    CSRvalue = (card->intnum << CON_VECSHIFT);
	
    /* Toggle the 'test' bit the requested passed 'count' number of times */ 
    for(iClockCount = count; iClockCount > 0; iClockCount--)
    {
        CSRvalue |= CSR_TEST;
        vxMemProbe((char *)(card->mem + CRG_CONTROL),	WRITE, sizeof(short), (char *) (&CSRvalue));
        CSRvalue &= ~CSR_TEST;
        vxMemProbe((char *)(card->mem + CRG_CONTROL),	WRITE, sizeof(short), (char *) (&CSRvalue));
	}

    /* Return No Errors */
	return(0);
}


#define IO_REPORT	DrvHy8513Report
#define DRIVER_INIT	Hy8513Init

/* Driver entry table */
struct
{
	long		number;
    long	    (*report)();
    long	    (*init)();
}DRVTAB=
{
	2,
    IO_REPORT,
    DRIVER_INIT
};

