/***************************************************************************

Project:
    CAN Bus Driver for EPICS

File:
    drvIpac.h

Description:
    IPAC Driver header file, defines the software interfaces:
	1. Upwards to the IPAC Module driver
	2. Downwards to the IPAC Carrier driver

Author:
    Andrew Johnson <anjohnson@iee.org>
Created:
    1 July 1995
Version:
    drvIpac.h,v 1.5 2000/02/21 21:35:33 anj Exp

Copyright (c) 1995-2000 Andrew Johnson

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*******************************************************************************/
/* modified by Scotty (walter.scott@acm.org), based on
   Andrew's version '1.5 2000/02/21 21:35:33 anj Exp'.
   Added hotswap ability but retained backwards compatability to
   other carrier cards.
   Please see the Hy8002 VME64X IP carrier card EPICS driver
   documentation on http://wwww.hytec-electronics.co.uk for
   all modified source code.

   ALL CHANGES IN THE FILE ARE COMMENTED BY *SCO*

   $Id: drvIpac.h,v 1.1.1.1 2004/06/04 14:16:30 cross Exp $
*/


#ifndef INCdrvIpacH
#define INCdrvIpacH

#include <types.h>

#ifndef NO_EPICS
#include <errMdef.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif


/* Error numbers */

#ifndef OK
#define OK 0
#endif

#ifndef M_ipac
#define M_ipac		(600 << 16)
#endif

#define S_IPAC_badTable   (M_ipac| 1) /*IPAC Carrier Table invalid*/
#define S_IPAC_tooMany    (M_ipac| 2) /*Too many IPAC carriers, table full*/
#define S_IPAC_badAddress (M_ipac| 3) /*Bad IPAC carrier or slot number*/
#define S_IPAC_badDriver  (M_ipac| 4) /*Bad value from IPAC carrier driver*/
#define S_IPAC_noModule   (M_ipac| 5) /*No IP module installed*/
#define S_IPAC_noIpacId   (M_ipac| 6) /*IPAC identifier not found*/
#define S_IPAC_badCRC     (M_ipac| 7) /*IPAC CRC Check failed*/
#define S_IPAC_badModule  (M_ipac| 8) /*IPAC Manufacturer or model ID wrong*/
#define S_IPAC_notImplemented (M_ipac| 9) /*IPAC Driver command not available*/
#define S_IPAC_badVector (M_ipac| 10) /*Bad interrupt vector*/
#define S_IPAC_vectorInUse (M_ipac| 11) /*Interrupt vector in use*/
#define S_IPAC_badIntLevel (M_ipac| 12) /*Bad interrupt level*/
/* *SCO* */
#define S_IPAC_mallocFailed (M_ipac| 13)/*memory allocation failed*/


/* Structure of the IPAC ID Prom, located in the pack ID space. */

typedef volatile struct {
    uint16_t asciiI;
    uint16_t asciiP;
    uint16_t asciiA;
    uint16_t asciiC;
    uint16_t manufacturerId;
    uint16_t modelId;
    uint16_t revision;
    uint16_t reserved;
    uint16_t driverIdLow;
    uint16_t driverIdHigh;
    uint16_t bytesUsed;
    uint16_t CRC;
    uint16_t packSpecific[52];
} ipac_idProm_t;


/* These are the types of address space implemented in the IP
   specification.  Some IP modules only use the ID and IO spaces. */

#define IPAC_ADDR_SPACES 4

typedef enum {
    ipac_addrID = 0,	/* ID Prom space */
    ipac_addrIO = 1,	/* Registers etc */
    ipac_addrIO32 = 2,	/* Registers for 32-bit dual-slot */
    ipac_addrMem = 3	/* Memory space */
} ipac_addr_t;


/* The following are the possible commands to the carrier driver to 
   handle configuration for the IP modules.  Most carriers will only be 
   able to implement a subset of these commands.  Note that irqEnable 
   should call the vxWorks sysBusEnable routine if this is needed to 
   pass the carrier interrupts through to the CPU. The ipac_stat commands
   were added for the VIPC664, and provide a means for showing the current
   status of each module using LEDs provided for each slot. */

typedef enum {
    ipac_irqLevel0 = 0,	/* Disables interrupts */
    ipac_irqLevel1 = 1,	/* Lowest priority */
    ipac_irqLevel2 = 2,	
    ipac_irqLevel3 = 3,	
    ipac_irqLevel4 = 4,	
    ipac_irqLevel5 = 5,	
    ipac_irqLevel6 = 6,	/* Highest priority */
    ipac_irqLevel7 = 7,	/* Non-maskable, don't use */
    ipac_irqGetLevel,	/* Returns level set (or hard-coded) */
    ipac_irqEnable,	/* Required to use interrupts, sets statActive */
    ipac_irqDisable,	/* Not necessarily supported */
    ipac_irqPoll,	/* Returns interrupt state */
    ipac_irqSetEdge,	/* Sets edge-triggered interrupts */
    ipac_irqSetLevel,	/* Sets level-triggered (default) */
    ipac_irqClear,	/* Only needed if using edge-triggered */
    ipac_statUnused,	/* Empty/uninitialized (Red LED on) */
    ipac_statActive	/* Slot in use (Green LED on) */
} ipac_irqCmd_t;


/* This is a table which each IPAC carrier driver provides to allow
   it to be queried by the IPAC driver.  One table is required for
   each type of carrier.  The cPrivate pointer is returned by the
   carrier driver initialise routine, and passed to all of the other
   routines as a means of identification of the carrier board. */

typedef struct {
    char *carrierType;
			/* String containing carrier board type */
    ushort_t numberSlots;
			/* Number of IPAC devices this carrier can hold */
    int (*initialise)(char *cardParams, void **cPrivate, ushort_t carrier);
			/* Initialise carrier and return *cPrivate */
    char *(*report)(void *cPrivate, ushort_t slot);
			/* Return string with giving status of this slot */
    void *(*baseAddr)(void *cPrivate, ushort_t slot, ipac_addr_t space);
			/* Return base addresses for this slot */
    int (*irqCmd)(void *cPrivate, ushort_t slot, 
		ushort_t irqNumber, ipac_irqCmd_t cmd);
			/* Interrupt manipulation */
    int (*intConnect)(void *cPrivate, ushort_t slot, ushort_t vecNum, 
		void (*routine)(int parameter), int parameter);
			/* Connect routine to interrupt vector */
} ipac_carrier_t;


/*these are the events that drvIpac layer will send to interested
  parties:

  a) a carrier driver providing a HotSwap function
     in its jump table.

  b) an IP driver installing a callback function with 
     ipmInstallHSCallBack().
*/

typedef enum HS_event{HS_Extract,HS_Insert}HS_event;



/* *SCO* add a new backwards compatible
   structure type, which can be better extended in future.
   funcnum will tell us how many functions the table has.
*/

typedef struct {
  char *carrierType;
    /* String containing carrier board type */

  ushort_t numberSlots;
  /* Number of IPAC devices this carrier can hold */

  ushort_t funcnum;
  /*the number of functions (>=5) the table contains.
    Five denotes a standard carrier without hotswap capability.
    Six or seven denotes a carrier with hotswap capability.
  */
  
  /****the functions, as defined above*/
  int (*initialise)(char *cardParams, void **cPrivate, ushort_t carrier);
  /* Initialise carrier and return *cPrivate */

  char *(*report)(void *cPrivate, ushort_t slot);
  /* Return string with giving status of this slot */

  void *(*baseAddr)(void *cPrivate, ushort_t slot, ipac_addr_t space);
  /* Return base addresses for this slot */

  int (*irqCmd)(void *cPrivate, ushort_t slot, 
		ushort_t irqNumber, ipac_irqCmd_t cmd);
  /* Interrupt manipulation */

  int (*intConnect)(void *cPrivate, ushort_t slot, ushort_t vecNum, 
		    void (*routine)(int parameter), int parameter);
  /* Connect routine to interrupt vector */

  /*add new functions here*/
  /* ******* THE carrierIsPresent AND HotSwap FUNCTIONS ARE NEVER
     CALLED BY THE IPAC LAYER (EVEN INDIRECTLY), ONLY THE
     DRVIPAC LAYER WILL CALL THEM IF PRESENT.
  ******/
  int (*carrierIsPresent)(void* cPrivate);
  /* This routine must probe the carrier card registers and
     return true (>0) if this carrier board is physically
     in the system. Return false (0) if it has been hotswapped out.
  */

  int (*HotSwap)(void* cPrivate, HS_event hsevent);
  /*the drvIpac layer will call this routine provided
    by the carrier when a hotswap event occurs.
    HotSwap(cp,HS_Extract);
    when the user has requested a scheduled hot swap.
    The carrier can prepare itself to be switched off.

    HotSwap(cp,HS_Insert);
    when the user has reinserted the carrier and scheduled
    the carrier to be reinstalled into the system.
    The carrier should resume its normal function.
  */

} EXTipac_carrier_t;



/* Functions for startup and interactive use */

/* this is the backwards compatible version of the routine
   -- not hotswap support*/
extern int ipacAddCarrier(ipac_carrier_t *pcarrier, char *cardParams);

  /* *SCO*
     USE THIS VERSION FOR CARRIER HOTSWAP SUPPORT
  */
  extern int ipacEXTAddCarrier(EXTipac_carrier_t *pcarrier, char *cardParams);
  extern int ipacReport(int interest);
  extern int ipacInitialise(int after);


  extern int ipacHSRemove(ushort_t carrier,int autoinsert);
  /* *SCO* the user will call this function before removing a
     carrier card from the VME crate.
     if autoinsert is nonzero, the carrier will come alive
     as soon as the carrier is reinserted,
     without the user having to type ipacHSInsert.
*/

  extern int ipacHSInsert(ushort_t carrier);
  /* *SCO* the user will call this function after having
     reinserted the carrier card into the VME crate*/



/* Functions for use in IPAC module drivers */

  extern int ipmCheck(ushort_t carrier, ushort_t slot);
  extern int ipmValidate(ushort_t carrier, ushort_t slot,
			 uchar_t manufacturerId, uchar_t modelId);
  extern char *ipmReport(ushort_t carrier, ushort_t slot);
  extern void *ipmBaseAddr(ushort_t carrier, ushort_t slot, ipac_addr_t space);
  extern int ipmIrqCmd(ushort_t carrier, ushort_t slot, 
		       ushort_t irqNumber, ipac_irqCmd_t cmd);
  extern int ipmIntConnect(ushort_t carrier, ushort_t slot, ushort_t vector, 
			   void (*routine)(int parameter), int parameter);

/* *SCO*
   ALLOWS THE IPAC DRIVER TO SEE WHETHER THE CARRIER HAS BEEN
   HOTSWAPPED OUT OR NOT.
*/
extern int ipmCarrierIsPresent(ushort_t carrier);
/* return true (>0) if this carrier board is physically
   in the system.
   return false (0) if it has been hotswapped out.
   return a negative number if carriernum is an invalid number.
*/




typedef int (*IPMHSFUNC)(ushort_t ipcardnum, HS_event hsevent);


/* *SCO*
   ALLOWS THE IPAC DRIVER TO INSTALL A CALLBACK ROUTINE OF TYPE
   IPMHSFUNC TO HANDLE HOTSWAP EVENTS.
*/
  extern int ipmInstallHSCallBack(ushort_t carrier,IPMHSFUNC hsfunc);



#ifdef __cplusplus
}
#endif

#endif /* INCipacH */

